package de.j4ee.util.csv;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import de.j4ee.util.CommonUtil;
import de.j4ee.util.Tracer;
import de.j4ee.util.csv.io.FullReader;
import de.j4ee.util.csv.io.LazyStreamReader;
import de.j4ee.util.csv.io.ReaderInterface;

public class CsvTokenizer
{
	private final static String WINDOWSLINEENDER = "\r\n";
	private final static String UNIXLINEENDER = "\n";
	private String actualLineEnder = System.getProperty("line.separator");
	private InputStream in = null;
	private String delimiter = null;
	private char delimiterChar = (char) -1;
	private boolean delimIsOneCharacter = false;
	private char currentChar = (char) -1;
	private long textLength = -1;
	private long currentPos = 0;
	private ReaderInterface reader = null;

	public CsvTokenizer(File file, String delimiter, boolean sequentialLoad) throws IOException
	{
		textLength = file.length();
		this.delimiter = delimiter;
		if (delimiter.length() == 1)
		{
			delimIsOneCharacter = true;
			delimiterChar = delimiter.charAt(0);
		}
		if (!sequentialLoad)
		{
			reader = new FullReader(file);
		}
		else
		{
			in = CommonUtil.getInputStream(file);
			reader = new LazyStreamReader(in);
		}
	}

	public CsvTokenizer(long length, InputStream is, String delimiter)
	{
		textLength = length;
		this.delimiter = delimiter;
		if (delimiter.length() == 1)
		{
			delimIsOneCharacter = true;
			delimiterChar = delimiter.charAt(0);
		}
		reader = new LazyStreamReader(is);
	}

	public CsvTokenizer(File file, String delimiter) throws IOException
	{
		this(file, delimiter, false);
	}

	public CsvTokenizer(String content, String delimiter) throws IOException
	{
		textLength = (int) content.length();
		this.delimiter = delimiter;
		if (delimiter.length() == 1)
		{
			delimIsOneCharacter = true;
			delimiterChar = delimiter.charAt(0);
		}
		{
			reader = new LazyStreamReader(new ByteArrayInputStream(content.getBytes()));
		}
	}

	public void reset()
	{
		reset(delimiter);
	}

	public void reset(String delim)
	{
		this.delimiter = delim;
		reader.reset();
	}

	private char getRelCharAt(int ahead)
	{
		if (ahead == 0)
		{
			return currentChar;
		}
		else
		{
			return reader.getRelativeChar(ahead);
		}
	}

	private void step(int step)
	{
		currentPos = reader.step(step);
		currentChar = reader.getCurrentChar();
	}

	private final static int Eof = -1;
	private final static int LineEnder = 0;
	private final static int Delimiter = 1;
	private final static int Quote = 2;
	private final static int Space = 3;
	private final static int Character = 4;

	private int getType()
	{
		return getType(0);
	}

	private int getType(int ahead)
	{
		char character = getRelCharAt(ahead);
		int type = Character;
		if (isDelim(ahead))
		{
			type = Delimiter;
		}
		else if (character == ' ')
		{
			type = Space;
		}
		else if (character == '"')
		{
			type = Quote;
		}
		else if (isLineBreak(ahead))
		{
			type = LineEnder;
		}
		else if (((byte) character) == -1)
		{
			type = Eof;
		}
		return type;
	}

	public String nextToken()
	{
		long tokenStartPos=currentPos;
		int quoteCounter = 0;
		StringBuffer token = new StringBuffer();
		step(0);
		while (textLength > currentPos)
		{
			int type = getType();
			if (type == Quote)
			{
				{
					int type2 = getType(1);
					if (type2 == Quote && currentPos!=tokenStartPos)
					{
						// 2 Quotes together
						token.append(currentChar);
						step(2);
						continue;
					}
					else if (type2 == Delimiter || type2 == LineEnder || type2==Eof)
					{
						// quote and correct delim/lf is followed
						--quoteCounter;
						step(1);
						continue;
					}
					else if (type2 == Space && isDelim(2))
					{
						// patch: quote followed by space and delim
						if (quoteCounter > 0)
							--quoteCounter;
						step(2);
						continue;
					}
					else
					{
						// 2
						++quoteCounter;
						step(1);
						continue;
					}
				}
			}
			else
			{
				//System.err.println(quoteCounter);
				if (type == Delimiter && quoteCounter <= 0)
				{
					step(delimiter.length());
					break;
				}
				else if (type == LineEnder && quoteCounter <= 0)
				{
					if (token.toString().trim().length() == 0)
					{
						token = null;
						step(actualLineEnder.length());
					}
					break;
				}
			}
			token.append(currentChar);
			step(1);
		}
		if (token != null)
			return token.toString();
		else
			return null;
	}

	private boolean isLineBreak(int ahead)
	{
		char character = getRelCharAt(ahead);
		if (character == WINDOWSLINEENDER.charAt(0))
		{
			setLineBreakType(WINDOWSLINEENDER);
			return IsRelText(ahead, 1, WINDOWSLINEENDER);
		}
		else if (character == UNIXLINEENDER.charAt(0))
		{
			setLineBreakType(UNIXLINEENDER);
			return true;
		}
		return false;
	}

	private void setLineBreakType(String nLB)
	{
		actualLineEnder = nLB;
	}

	private boolean isDelim(int ahead)
	{
		char character = getRelCharAt(ahead);
		if (character != delimiterChar && delimIsOneCharacter)
			return false;
		else
			return IsRelText(ahead, 1, delimiter);
	}

	private boolean IsRelText(int ahead, int off, String text)
	{
		int textl = text.length();
		boolean ret = true;
		for (int counter = off; ret && counter < textl; counter++)
			ret = getRelCharAt(ahead + counter) == text.charAt(counter);
		return ret;
	}

	public boolean hasMoreToken()
	{
		boolean ret = (textLength > currentPos);
		return ret;
	}

	public void close()
	{
		if (reader != null)
		{
			reader.close();
		}
		if (in != null)
		{
			try
			{
				in.close();
			}
			catch (IOException e)
			{
				Tracer.printThrowable(e);
			}
		}
	}
}